import numpy as np
import matplotlib.pyplot as plt
from scipy import signal

# ---------- Setup ----------
np.random.seed(0)

fs = 500.0          # Hz
T  = 4.0            # seconds
t  = np.arange(0, T, 1/fs)

# Test signal: slow trend + mid tone + noise
x = 0.6*np.sin(2*np.pi*0.7*t) + 0.8*np.sin(2*np.pi*10*t) + 0.35*np.random.randn(t.size)

# EMA parameters to compare
alphas = [0.05, 0.2, 0.7]

# ---------- Implementation 1: hand-coded recursion ----------
def ema_hand(x, alpha):
    y = np.zeros_like(x)
    y_prev = 0.0  # zero-state init; use x[0] for steady start if preferred
    a = 1.0 - alpha
    for n, xn in enumerate(x):
        y_curr = alpha*xn + a*y_prev
        y[n] = y_curr
        y_prev = y_curr
    return y

# ---------- Implementation 2: SciPy lfilter ----------
def ema_scipy(x, alpha):
    b = np.array([alpha], dtype=float)            # numerator
    a = np.array([1.0, -(1.0 - alpha)], float)    # denominator: 1 - (1-a) z^-1
    # lfilter uses zero initial conditions by default (matches hand-coded above)
    return signal.lfilter(b, a, x)

# ---------- Apply filters ----------
ys_hand  = {a: ema_hand(x, a)  for a in alphas}
ys_scipy = {a: ema_scipy(x, a) for a in alphas}

# ---------- Plot: time-domain smoothing ----------
plt.figure(figsize=(8, 4.2))
plt.plot(t, x, linewidth=1.0, alpha=0.5, label="input x[n]")
for a in alphas:
    plt.plot(t, ys_hand[a], linewidth=1.2, linestyle='--', label=f"EMA hand (α={a})")
plt.xlabel("Time [s]")
plt.ylabel("Amplitude")
plt.title("Exponential Moving Average (EMA) - Time Domain")
plt.xlim(0, 1)
plt.grid(True, alpha=0.35)
plt.legend(ncol=2, fontsize=9)
plt.tight_layout()
plt.show()

# Optional overlay to verify equivalence (hand vs SciPy) on the largest α
a_check = alphas[-1]
err = np.max(np.abs(ys_hand[a_check] - ys_scipy[a_check]))
print(f"Max |hand - scipy| for α={a_check}: {err:.3e}")

# ---------- Plot: frequency response for the three α ----------
plt.figure(figsize=(8, 4.2))
w = np.linspace(0, np.pi, 4096)
for a in alphas:
    b = np.array([a], float)
    A = np.array([1.0, -(1.0 - a)], float)
    w_resp, h = signal.freqz(b, A, worN=w)
    f_hz = (w_resp/(2*np.pi))*fs
    mag_db = 20*np.log10(np.maximum(np.abs(h), 1e-12))
    plt.plot(f_hz, mag_db, linewidth=1.2, label=f"α={a}")
plt.xlabel("Frequency [Hz]")
plt.ylabel("Magnitude [dB]")
plt.title("EMA Frequency Response |H(e^{jΩ})|")
plt.grid(True, alpha=0.35)
plt.ylim(-60, 5)
plt.legend(title="Parameter", fontsize=9)
plt.tight_layout()
plt.show()
