import numpy as np
import matplotlib.pyplot as plt
from scipy import signal

# --- Parameters ---
fs = 500.0             # Hz
T = 3.0                # seconds
t = np.arange(0, T, 1/fs)

# Test signal: low tone + mid tone + noise
rng = np.random.default_rng(0)
x = 0.7*np.sin(2*np.pi*3*t) + 0.8*np.sin(2*np.pi*40*t) + 0.35*rng.standard_normal(t.size)

# FIR coefficients (example: 21-tap moving average low-pass)
numtaps = 21
b = np.ones(numtaps, dtype=float) / numtaps
a = np.array([1.0], dtype=float)  # FIR => denominator is 1

# --- Implementation 1: by hand (streaming, causal) ---
def fir_by_hand(x, b):
    M = len(b) - 1
    y = np.zeros_like(x, dtype=float)
    # causal linear convolution with a circular buffer
    X = np.zeros(M+1, dtype=float)
    idx = 0
    for n, xn in enumerate(x):
        X[idx] = xn
        acc = 0.0
        j = idx
        for k in range(M+1):
            acc += b[k] * X[j]
            j = (j - 1) % (M+1)
        y[n] = acc
        idx = (idx + 1) % (M+1)
    return y

y_hand = fir_by_hand(x, b)

# --- Implementation 2: SciPy lfilter ---
y_lf = signal.lfilter(b, a, x)

# --- Verify equivalence ---
max_err = np.max(np.abs(y_hand - y_lf))

# --- Plot 1: Time-domain comparison ---
plt.figure(figsize=(8, 4.2))
plt.plot(t, x, linewidth=0.9, alpha=0.5, label="Input x[n]")
plt.plot(t, y_hand, linewidth=1.4, label=f"FIR by hand (numtaps={numtaps})")
plt.plot(t, y_lf, linestyle='--', linewidth=1.1, label="FIR via lfilter")
plt.title(f"FIR Filtering (Moving Average {numtaps}-tap) — Time Domain\\nmax|hand-lfilter|={max_err:.2e}")
plt.xlabel("Time [s]")
plt.ylabel("Amplitude")
plt.grid(True, alpha=0.35)
plt.legend(fontsize=9, ncol=2)
plt.tight_layout()
plt.show()

# --- Plot 2: Frequency response of the FIR ---
w, h = signal.freqz(b, a, worN=4096)
f_hz = (w/(2*np.pi))*fs
mag_db = 20*np.log10(np.maximum(np.abs(h), 1e-12))
phase = np.unwrap(np.angle(h))

plt.figure(figsize=(8, 4.2))
plt.plot(f_hz, mag_db, linewidth=1.4, label="|H(e^{jΩ})| [dB]")
plt.title("FIR Frequency Response (Moving Average)")
plt.xlabel("Frequency [Hz]")
plt.ylabel("Magnitude [dB]")
plt.ylim(-80, 5)
plt.grid(True, alpha=0.35)
plt.tight_layout()
plt.show()

plt.figure(figsize=(8, 4.2))
plt.plot(f_hz, phase, linewidth=1.2, label="Phase")
plt.title("FIR Frequency Response — Phase (Unwrapped)")
plt.xlabel("Frequency [Hz]")
plt.ylabel("Phase [rad]")
plt.grid(True, alpha=0.35)
plt.tight_layout()
plt.show()
