import numpy as np
import matplotlib.pyplot as plt
from scipy import signal

# ---------- Build a synthetic test image (grayscale, [0,1]) ----------
H, W = 256, 256
yy, xx = np.mgrid[0:H, 0:W]
grad = (xx / (W - 1)) * 0.5  # horizontal gradient 0..0.5

# Checkerboard pattern
tile = 16
checker = (((yy // tile) + (xx // tile)) % 2).astype(float) * 0.4  # 0 or 0.4

# Concentric circle pattern
r = np.sqrt((xx - W/2)**2 + (yy - H/2)**2)
rings = 0.2 * (np.sin(2*np.pi * r / 20) * 0.5 + 0.5)  # soft rings

# Combine and add noise
rng = np.random.default_rng(0)
noise = 0.08 * rng.standard_normal((H, W))
img = np.clip(0.2 + grad + checker + rings + noise, 0.0, 1.0)

# Plot original
plt.figure(figsize=(4.5, 4.5))
plt.imshow(img, cmap='gray', vmin=0, vmax=1)
plt.axis('off')
plt.title("Original (synthetic)")
plt.show()

# ---------- Low-pass filtering: 2D box blur (9x9) ----------
k = 9
box = np.ones((k, k), dtype=float) / (k*k)
img_lp = signal.convolve2d(img, box, mode='same', boundary='symm')

plt.figure(figsize=(4.5, 4.5))
plt.imshow(img_lp, cmap='gray', vmin=0, vmax=1)
plt.axis('off')
plt.title(f"Low-pass blur (box {k}×{k})")
plt.show()

# ---------- High-pass (edges): Sobel magnitude ----------
sobel_x = np.array([[1, 0, -1],
                    [2, 0, -2],
                    [1, 0, -1]], dtype=float)
sobel_y = sobel_x.T
gx = signal.convolve2d(img, sobel_x, mode='same', boundary='symm')
gy = signal.convolve2d(img, sobel_y, mode='same', boundary='symm')
edge_mag = np.hypot(gx, gy)
edge_mag /= (edge_mag.max() + 1e-12)

plt.figure(figsize=(4.5, 4.5))
plt.imshow(edge_mag, cmap='gray', vmin=0, vmax=1)
plt.axis('off')
plt.title("High-pass edges (Sobel magnitude)")
plt.show()

# ---------- Sharpening via unsharp masking ----------
amount = 1.0
sharp = np.clip(img + amount * (img - img_lp), 0.0, 1.0)

plt.figure(figsize=(4.5, 4.5))
plt.imshow(sharp, cmap='gray', vmin=0, vmax=1)
plt.axis('off')
plt.title(f"Sharpened (unsharp, amount={amount})")
plt.show()
