#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <omp.h>

typedef struct node
{
	int val;
	struct node *left;
	struct node *right;
} node;


node *create(int level)
{
	static int gx = 1;
	node *p;
	int x;

	if (level==4)
		return NULL;

	x = gx++;

	p=(node*)malloc(sizeof(node));
	p->val=x;

	printf("Enter left child of %d:\n",x);
	p->left=create(level+1);

	printf("Enter right child of %d:\n",x);
	p->right=create(level+1);

	return p;
}

int _print_t(node *tree, int is_left, int offset, int depth, char s[20][255])
{
    char b[20];
    int width = 5;

    if (!tree) return 0;

    sprintf(b, "(%03d)", tree->val);

    int left  = _print_t(tree->left,  1, offset,                depth + 1, s);
    int right = _print_t(tree->right, 0, offset + left + width, depth + 1, s);

    for (int i = 0; i < width; i++)
        s[2 * depth][offset + left + i] = b[i];

    if (depth && is_left) {

        for (int i = 0; i < width + right; i++)
            s[2 * depth - 1][offset + left + width/2 + i] = '-';

        s[2 * depth - 1][offset + left + width/2] = '+';
        s[2 * depth - 1][offset + left + width + right + width/2] = '+';

    } else if (depth && !is_left) {

        for (int i = 0; i < left + width; i++)
            s[2 * depth - 1][offset - width/2 + i] = '-';

        s[2 * depth - 1][offset + left + width/2] = '+';
        s[2 * depth - 1][offset - width/2 - 1] = '+';
    }

    return left + width + right;
}

void print_tree(node *tree)
{
    char s[20][255];
    for (int i = 0; i < 20; i++)
        sprintf(s[i], "%80s", " ");

    _print_t(tree, 0, 0, 0, s);

    for (int i = 0; i < 20; i++)
        printf("%s\n", s[i]);
}

void preorder(node *t)
{
	if(t!=NULL)
	{
		printf("%d\n",t->val);
		preorder(t->left);
		preorder(t->right);
	}
}

int get_node_val(node *t)
{
	sleep(1);
	printf("[%d] getting node val: %d\n", omp_get_thread_num(), t->val);
	return t->val;
}

int process(node *t)		//address of root node is passed in t
{
	if (t == NULL)
		return 0;


	if ((t->left == NULL) && (t->right == NULL))
	{
		int val = get_node_val(t);
		return val;
	}
	else
	{
		int lsum=0, rsum=0;

		if (t->left)
			lsum = process(t->left);

		if (t->right)
			rsum = process(t->right);

		int val = get_node_val(t);

		return lsum+rsum+val;
	}
}

int process_par(node *t)		//address of root node is passed in t
{
	if (t == NULL)
		return 0;

	if ((t->left == NULL) && (t->right == NULL)) {
		int val = get_node_val(t);
		return val;
	}
	else {
		int lsum=0, rsum=0;

		if (t->left) {
			#pragma omp task shared(lsum)
			lsum = process_par(t->left);
		}

		if (t->right) {
			#pragma omp task shared(rsum)
			rsum = process_par(t->right);
		}

		#pragma omp taskwait
		int val = get_node_val(t);

		return lsum+rsum+val;
	}
}


int main()
{
	node *root;
	root=create(0);
	printf("\nThe preorder traversal of tree is:\n");
	preorder(root);

	print_tree(root);

	double t0, t1;
	int sum;

	t0 = omp_get_wtime();
	sum = process(root);
	t1 = omp_get_wtime();
	printf("seq sum=%d in %.2lf s\n", sum, t1-t0);

	t0 = omp_get_wtime();
	#pragma omp parallel num_threads(2)
	{
	#pragma omp single nowait
	{
	sum = process_par(root);
	}
	}

	t1 = omp_get_wtime();

	printf("par sum=%d in %.2lf s\n", sum, t1-t0);

	return 0;
}
