// Description: Compute eigenvalues of Lehmer matrix
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <omp.h>

// interface for LAPACK routines.
#include <lapack.h>

// macro for the lehmer matrix
#define lehmer(i,j) ((i) <= (j))? ((i)/(j)):((j)/(i))


///////////////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
    int N = 4096;  // default matrix size (NxN)

    for(int i = 1; i < argc; i++ ) {
        if( strcmp( argv[i], "-N" ) == 0 ) {
            N = atoi(argv[i+1]);
            i++;
        }
    }

    // Lehmer matrix
    int nrows = N, ncols = N;
    double *C = malloc(nrows*ncols*sizeof(double));

    // matrix initialization
    double ti1 = omp_get_wtime();

#if defined(_OPENMP)
    #pragma omp parallel for schedule(guided)
#endif
    for (int i = 0; i < nrows; i++) {
        for (int j = 0; j <= i; j++) {
            const double ii = i+1;
            const double jj = j+1;
            C[j*nrows + i] = lehmer(ii,jj); // column-major layout
        }
    }

    double ti2 = omp_get_wtime();
    printf("Init time = %lf seconds\n", ti2-ti1);

    // Study the function signature for the dsyev_() routine:
    // http://www.netlib.org/lapack/explore-html/d2/d8a/group__double_s_yeigen_ga442c43fca5493590f8f26cf42fed4044.html#ga442c43fca5493590f8f26cf42fed4044

    // required parameters for the dsyev_() routine
    char jobz = 'N'; // we are only interested in the eigenvalues
    // Note: We have initialized the lower-triangular part of C, we must choose
    // L here.
    char uplo = 'L';

    // first call to dsyev_() with lwork = -1 to determine the optimal
    // workspace.  This does not yet compute the eigenvalues but determines the
    // optimal workset.
    double *work, *W;
    W = malloc(N*sizeof(double));
    assert(W != NULL);
    work = malloc(1*sizeof(double));
    assert( work != NULL);

    int info, lwork;
    lwork=-1;

    dsyev_(&jobz, &uplo, &N, C, &N, W, work, &lwork, &info); // 1, 1);

    lwork= (int) work[0];
    free(work);

    // prepape and issue the second (actual) call.
    work = malloc(lwork*sizeof(double));
    assert(work != NULL);

    double t1 = omp_get_wtime();

    dsyev_(&jobz, &uplo, &N, C, &N, W, work, &lwork, &info); // 1, 1);

    double t2 = omp_get_wtime();

    // Output results
    printf("Elapsed time = %lf seconds\n", t2-t1);
    printf("Largest Eigenvalues:\n");
    for (int i = N-1; i >= N-6; i--) {
        printf("%.3lf\n", W[i]);
    }

    free(work);
    free(W);
    free(C);

    return 0;
}
